/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2016 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for CertificateServiceRequest
 */

var CardActionWizard = require('pki-as-a-service/ui/CardActionWizard').CardActionWizard;
var SubjectDAO = require('scsh/pki-db/db/SubjectDAO').SubjectDAO;
var I18N = require('scsh/srv-cc1/I18N').I18N;
var PKIAPIConnector = require('scsh/srv-cc1/PKIAPIConnector').PKIAPIConnector;
var PKIAPICardActionWizard = require('pki-as-a-service/ui/PKIAPICardActionWizard').PKIAPICardActionWizard;



I18N.addResources([
	{ lang: "EN", keys: {
		"action.dfncr.retrieve": "Retrieve Certificate",
		"action.dfncr.forward": "Forward Application",
		"action.dfncr.reject": "Reject Application",

		"msg.dfncr.title": "Request DFN Certificate",
		"msg.dfncr.applicant": "Certificate Holder Information",
		"msg.dfncr.ra": "Registration Authority",
		"msg.dfncr.signerId": "RA-ID",
		"msg.dfncr.externalApproval": "External Approval",

		"msg.dfncr.email": "Email",
		"msg.dfncr.name": "Name",
		"msg.dfncr.orgUnit": "Department",

		"msg.dfncr.subjectAlternativeName": "Subject Alternative Names",
		"msg.dfncr.dns": "DNS",
		"msg.dfncr.ip": "IP",
		"msg.dfncr.uri": "URI",
		"msg.dfncr.mupn": "Microsoft UPN",

		"msg.dfncr.request": "Certificate Request Information",
		"msg.dfncr.role": "Role",
		"msg.dfncr.pin": "Revocation Password",
		"msg.dfncr.prefix": "DN Prefix",

		"msg.dfncr.identityVerification": "Identity Verification",
		"msg.dfncr.internalReviewRemark": "Internal Remarks",
	}},
	{ lang: "DE", keys: {
		"action.dfncr.retrieve": "Zertifikat abrufen",
		"action.dfncr.forward": "Antrag weiterleiten",
		"action.dfncr.reject": "Antrag ablehnen",

		"msg.dfncr.title": "DFN Zertifikatsantrag",
		"msg.dfncr.applicant": "Antragsteller",
		"msg.dfncr.ra": "Registrierungsstelle",
		"msg.dfncr.signerId": "RA-ID",
		"msg.dfncr.externalApproval": "Extern genehmigt",

		"msg.dfncr.email": "Email",
		"msg.dfncr.name": "Name",
		"msg.dfncr.orgUnit": "Abteilung",

		"msg.dfncr.subjectAlternativeName": "Weitere Namen",
		"msg.dfncr.dns": "DNS",
		"msg.dfncr.ip": "IP",
		"msg.dfncr.uri": "URI",
		"msg.dfncr.mupn": "Microsoft UPN",

		"msg.dfncr.request": "Zertifikatsrequest",
		"msg.dfncr.role": "Rolle",
		"msg.dfncr.pin": "Sperrkennwort",
		"msg.dfncr.prefix": "DN Präfix",

		"msg.dfncr.identityVerification": "Identitätsfeststellung",
		"msg.dfncr.internalReviewRemark": "Interner Vermerk",
	}}
]);



function DFNCertificateRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new DFNCertificateRequestController()");
}

exports.DFNCertificateRequestController = DFNCertificateRequestController;





DFNCertificateRequestController.prototype.createRequest = function(srct, parameter) {

	var trustCenterId = parseInt(parameter.id);
	GPSystem.log(GPSystem.DEBUG, module.id, "Selected TrustCenter ID:" + trustCenterId);

	var trustCenter = srct.service.getSubject(trustCenterId);

	if (trustCenter == null) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Can't find trust center");
	}

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "DFNCertificateServiceRequest",
		title: srct.i18n("msg.dfncr.title"),
		originatorId: srct.ui.session.user.id,
		recipientId: trustCenter.bo.id,
		state: "New request",
		content: {
		}
	};

	if (!trustCenter.checkServiceRequestCreation(srct.ui.session.user, t)) {
		return undefined;
	}

	if (trustCenter.canRead(srct.ui.session.user) || trustCenter.isManagerOf(srct.ui.session.user)) {
		var bo = srDAO.newServiceRequest(t);
	} else {
		return undefined;
	}

	return bo;
}



DFNCertificateRequestController.prototype.createInitializedRequest = function(service, template) {
	var srDAO = service.daof.getServiceRequestDAO();

	var t = {
		parentServiceRequestId: template.parent,
		process: "DFNCertificateServiceRequest",
		title: I18N.t(template.lang, "msg.cr.title"),
		details: template.commonName,
		originatorId: template.originatorId,
		recipientId: template.recipientId,
		state: "Submit",
		lifecycle: '3',
		assignedToRole: template.assignedToRole,
		content: {
			commonName: template.commonName,
			role: template.assignedToRole,
		}
	};

	var bo = srDAO.newServiceRequest(t);
	var sr = service.getServiceRequestById(bo.id);

	sr.transformSubjectDN();
	sr.commit();

	return bo;
}



DFNCertificateRequestController.prototype.postCreateAction = function(req, res, srct, action) {
	var bo = this.createRequest(srct, req.parameters);

	srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
	return true;
}



DFNCertificateRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {

	// REST call to the DFN Connector
	// - action requestCertificate
	// - action issueCertificate
	switch(action) {
	case "action.save":
	case "action.submit":
	case "action.dfncr.reject":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	case "action.cr.reqgen.usertoken":
	case "action.cr.publish.usertoken":
		srct.ui.wizard = new CardActionWizard(srct.ui, sr, "/" + req.url[1] + "/" + req.url[2]);
		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	case "action.cr.requestRevocation":
	case "action.dfncr.retrieve":
	case "action.dfncr.forward":
		var cfg = srct.service.dfnPKICfg;
		var data = { user: srct.ui.session.user.id, action: action, args: { fields: req.parameters } };
		var api = new PKIAPIConnector(cfg);
		api.postAction(sr.getId(), data);
		return false;
	case "action.cr.revoke":
	case "action.approve":
		var cfg = srct.service.dfnPKICfg;
		var api = new PKIAPIConnector(cfg);
		var data = { user: srct.ui.session.user.id, action: action, args: { fields: req.parameters } };
		var redirectUrl = "/" + req.url[1] + "/" + req.url[2];
		srct.ui.wizard = new PKIAPICardActionWizard(srct.ui, api, sr.getId(), data, cfg.rtURL, redirectUrl);

		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



DFNCertificateRequestController.prototype.getCreateView = function(req, res, srct) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div =	<div>
				<h1>Request a Certificate</h1>
				<p>This service will request a new certificate from one of the registered
				   certification authorities.</p>
			</div>

	var form =
		<form class="pure-form" action="" method="post" enctype="multipart/form-data">
			<p>Please select a certification authority from which you would like to receive a certificate.</p>
			<select name="id"/>
			<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
		</form>

	var subjectDAO = srct.ui.service.daof.getSubjectDAO();
	var l = form.select;
	var hasEntries = false;
	var trustCenters = subjectDAO.listSubjectsByType(SubjectDAO.TYPE_TRUST_CENTER);
	for each (var subject in trustCenters) {
		var trustCenter = srct.service.getSubject(subject.id);
		if ((trustCenter.canRead(srct.ui.session.user) || trustCenter.isManagerOf(srct.ui.session.user))
			 && (typeof(trustCenter.bo.getContent().dfnpkiRA) != "undefined")) {
			l.appendChild(<option value={subject.id}>{subject.name}</option>);
			hasEntries = true;
		}
	}

	if (hasEntries) {
		div.appendChild(<p>The service request consists of the following steps:</p>);
		div.appendChild(
			<ol>
				<li>You select from which certification authority you would like to receive a certificate</li>
				<li>You provide informations required by the certification authority</li>
				<li>A key pair is generated on your SmartCard-HSM</li>
				<li>A certificate request is generated and forwarded to a registration officer of the CA</li>
				<li>The registration officer either approves or rejects your request</li>
				<li>If the request is approved, then it is forwarded to the certification officer of the CA</li>
				<li>The certification officer will issue the certificate</li>
				<li>In the last step the certificate is stored on your SmartCard-HSM</li>
			</ol>);
		div.appendChild(form);
	} else {
		div.appendChild(<p>There is no accessible certification authority. This is either the case if there are no operational certification authorities or you are not authorized to access a private certification authority.</p>);
	}
	return div;
}



DFNCertificateRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	if (sr.hasCertificates() > 0) {
		div.appendChild( <br/> );
		div.appendChild( <h2>Certificates</h2> );
		div.appendChild(srct.renderCertificateList(req, res, sr));
	}

	return div;
}
