/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for creating a DFN-PKI Registration Authority
 */

var TrustCenter = require('pki-as-a-service/subjects/TrustCenter').TrustCenter;
var RoleDAO = require('scsh/pki-db/db/RoleDAO').RoleDAO;
var CommonUI = require('scsh/srv-cc1/CommonUI').CommonUI;
var I18N = require('scsh/srv-cc1/I18N').I18N;
var PKIAPIConnector = require('scsh/srv-cc1/PKIAPIConnector').PKIAPIConnector;
var PKIAPICardActionWizard = require('pki-as-a-service/ui/PKIAPICardActionWizard').PKIAPICardActionWizard;



I18N.addResources([
	{ lang: "EN", keys: {
		"action.createra": "Create",

		"msg.cdra.title": "Create DFN-PKI Registration Authority",

		"msg.cdra.subjectData": "DFNPKI Data",
		"msg.cdra.name": "CA Installation Name",
		"msg.cdra.raId": "RA-ID",
		"msg.cdra.keyStore": "RA-Keystore",
		"msg.cdra.description": "Description",

	}}
]);



function CreateDFNPKIRegistrationAuthorityRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new CreateDFNPKIRegistrationAuthorityRequestController()");
}

exports.CreateDFNPKIRegistrationAuthorityRequestController = CreateDFNPKIRegistrationAuthorityRequestController;



CreateDFNPKIRegistrationAuthorityRequestController.prototype.createRequest = function(srct, trustCenterId) {

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "CreateDFNPKIRegistrationAuthorityServiceRequest",
		title: srct.i18n("msg.cdra.title"),
		originatorId: srct.ui.session.user.id,
		recipientId: trustCenterId,
		state: "Created",
		content: {
		}
	};

	var bo = srDAO.newServiceRequest(t);
	return bo;
}



CreateDFNPKIRegistrationAuthorityRequestController.prototype.getSubject = function(req, res, srct) {
	var parameter = CommonUI.parseQueryString(req.queryString);
	var ca = Number(parameter.ca);

	if (isNaN(ca) || ca == 0) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Argument ca not valid in URL");
	}

	var subject = srct.service.getSubject(ca);

	if (!(subject instanceof TrustCenter)) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Subject is not a TrustCenter instance");
	}

	return subject;
}



CreateDFNPKIRegistrationAuthorityRequestController.prototype.postCreateAction = function(req, res, srct, action) {
	if (srct.ui.session.user.roles.indexOf(RoleDAO.ID_Subscriber) >= 0) {
		var parameter = CommonUI.parseQueryString(req.queryString);

		var trustCenter = this.getSubject(req, res, srct);

		if (!trustCenter.isManager(srct.ui.session.user)) {
			return false;
		}

		var bo = this.createRequest(srct, trustCenter.getId());

		srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
		return true;
	}

	return false;
}



CreateDFNPKIRegistrationAuthorityRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {

	switch(action) {
	case "action.submit":
	case "action.save":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	case "action.createra":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		var cfg = srct.service.dfnPKICfg;
		var api = new PKIAPIConnector(cfg);
		var data = { user: srct.ui.session.user.id, action: action, args: { } };
		var redirectUrl = "/" + req.url[1] + "/" + req.url[2];
		srct.ui.wizard = new PKIAPICardActionWizard(srct.ui, api, sr.getId(), data, cfg.rtURL, redirectUrl);

		var callback = function() {
			GPSystem.log(GPSystem.DEBUG, module.id, "Update the session's user roles");
// 			GPSystem.log(GPSystem.DEBUG, module.id, "User roles before: " + srct.ui.session.user.roles);
			var roleDAO = srct.service.daof.getRoleDAO();
			srct.ui.session.user.roles = roleDAO.getAssignedRoleIds(srct.ui.session.user.id);
// 			GPSystem.log(GPSystem.DEBUG, module.id, "User roles after: " + srct.ui.session.user.roles);
		}

		srct.ui.wizard.setCompletionHandler(callback);

		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



CreateDFNPKIRegistrationAuthorityRequestController.prototype.getCreateView = function(req, res, srct) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var form =	<div>
				<h1>Create a DFN-PKI Registration Authority</h1>

				<p>The creating TrustCenter will become a DFN-PKI RA role for approving certificate requests. The TrustCenter's existing RA
				will act as Basis-KeyUser</p>

				<p>Other TrustCenter can get the Basis-KeyUserRole for this RA via a JoinDFNPKIRegistrationAuthority service request.</p>
			</div>

	var trustCenter = this.getSubject(req, res, srct);

	if (trustCenter.isManager(srct.ui.session.user)) {
		form.appendChild(
			<form class="pure-form" action="" method="post" enctype="multipart/form-data">
					<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
			</form>
		);
	} else {
		form.appendChild(
			<p>You need the manager role to create a DFN-PKI Registration Authority for this Trust Center.</p>
		)
	}

	return form;
}



CreateDFNPKIRegistrationAuthorityRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	return div;
}
