/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2006 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Plug-In
 */

var SmartCardHSMKeySpecGenerator = require('scsh/sc-hsm/SmartCardHSM').SmartCardHSMKeySpecGenerator;
var File = require("scsh/file/File").File;



/**
 * Create a plug-in instance
 *
 * @constructor
 * @class A plug-in example
 * @param {KeyManager] km the associated key manager
 */
function ClonePlugin(km) {
	this.km = km;
}

// All plug-ins must export a symbol "Plugin"
exports.Plugin = ClonePlugin;

ClonePlugin.COPY_KEYS = "Copy Keys from Management Token";



/**
 * Add an entry to the context menu associated with the device entry in the outline
 *
 * @param {String[]} contextMenu the list of entries in the context menu
 * @param {Boolean} isInitialized the device is initialized
 * @param {Number} authenticationState the status returned in the last authentication query (SW1/SW2)
 */
ClonePlugin.prototype.addDeviceContextMenu = function(contextMenu, isInitialized, authenticationState) {
	if (this.km.managementToken && (authenticationState == 0x9000)) {
		contextMenu.push(ClonePlugin.COPY_KEYS);
	}
}



/**
 * Handle action triggered from the outline context menu
 *
 * @param {Outline} source the source outline node of this action
 * @param {String} action the action selected from the context menu
 * @type Boolean
 * @return true if the action was handled
 */
ClonePlugin.prototype.actionListener = function(source, action) {
	switch(action) {
		case ClonePlugin.COPY_KEYS:
			this.copyKeys(source);
			return true;
	}
	return false;
}



/**
 * Enumerate all key domains
 *
 * @param {SmartCardHSM} sc the token to look at
 * @type Object[]
 * @return the key domain status
 */
ClonePlugin.prototype.enumerateKeyDomains = function(sc) {
	var kd = [];
	var kdidx = -1;
	do {
		kdidx++;
		var kds = sc.queryKeyDomainStatus(kdidx);
		if (kds.sw == 0x9000) {
			kds.kdidx = kdidx;
			kd.push( kds  );
		}
	} while ((kds.sw != 0x6A86) && (kds.sw !== 0x6D00));

	return kd;
}



/**
 * Copy keys from the management token into the target token for all identical key domains
 *
 * @param {OutlineNode} the key domain outline node
 */
ClonePlugin.prototype.copyKeys = function(kd) {
	if (this.km.managementToken == undefined) {
		print("No management token found. Please insert management token and authenticate,");
		print("then select 'Use as management token' from the device context menu");
		return;
	}
	var mks = this.km.managementToken.ks;
	var tks = this.km.ks;

	var mkd = this.enumerateKeyDomains(mks.sc);
	var tkd = this.enumerateKeyDomains(tks.sc);

	// Create a list of XKEK key domains that are present on the master and the target
	var ckd = [];
	for (var i = 0; i < tkd.length; i++) {
		if (tkd[i].keyDomain) {
			for (var j = 0; j < mkd.length; j++) {
				if (mkd[j].keyDomain && mkd[j].keyDomain.equals(tkd[i].keyDomain)) {
					print("Found XKEK Key Domain " + tkd[i].keyDomain.toString(HEX));
					ckd.push( { m: mkd[j], t: tkd[i] } );
				}
			}
		}
	}

	// Establish a XKEK for all identical key domains
	var dom = new Key();
	dom.setComponent(Key.ECC_CURVE_OID, new ByteString("brainpoolP256r1", OID));
	for (var i = 0; i < ckd.length; i++) {
		var spec = new SmartCardHSMKeySpecGenerator(Crypto.EC, dom);

		spec.setKeyDomain(ckd[i].m.kdidx);
		spec.setAlgorithms(ByteString.valueOf(SmartCardHSM.ALG_ECDHXKEK));

		print("Generating exchange keys...");
		var label = "ex";
		if (mks.hasKey(label)) {
			mks.deleteKey(label);
		}
		var mreq = mks.generateKeyPair(label, spec);
		var mkey = mks.getKey(label);

		spec.setKeyDomain(ckd[i].t.kdidx);
		if (tks.hasKey(label)) {
			tks.deleteKey(label);
		}
		var treq = tks.generateKeyPair(label, spec);
		var tkey = tks.getKey(label);

		print("Key agreement...");
		mks.sc.verifyCertificate(this.km.certchain.dica);
		mks.sc.verifyCertificate(this.km.certchain.devicecert);
		mks.sc.deriveXKEK(mkey.getId(), treq, new ByteString("DerivationParam", ASCII));
		mks.deleteKey(label);

		tks.sc.verifyCertificate(this.km.managementToken.certchain.dica);
		tks.sc.verifyCertificate(this.km.managementToken.certchain.devicecert);
		tks.sc.deriveXKEK(tkey.getId(), mreq, new ByteString("DerivationParam", ASCII));
		tks.deleteKey(label);
	}

	var mkd = this.enumerateKeyDomains(mks.sc);
	var tkd = this.enumerateKeyDomains(tks.sc);

	// Create a list of key domains with the same KEK
	var ckd = [];
	for (var i = 0; i < tkd.length; i++) {
		for (var j = 0; j < mkd.length; j++) {
			if (tkd[i].kcv.equals(mkd[j].kcv)) {
				ckd[mkd[j].kdidx] = mkd[j];
			}
		}
	}

	var list = mks.enumerateKeys();

	// Find all keys that in matching key domains and that are not yet present on the target
	for (var i = 0; i < list.length; i++) {
		var key = mks.getKey(list[i]);
		if (ckd[key.keyDomain]) {
			if (key.algorithms != undefined && key.algorithms.find(ByteString.valueOf(SmartCardHSM.ALG_WRAP, 1)) < 0) {
				continue;
			}
			if (!tks.hasKey(key.getLabel())) {
				print("Transfering key '" + key.getLabel() + "'...");
				var keyblob = mks.exportKey(key);
				var newkey = tks.importKey(keyblob);

				tks.sc.updateKey(newkey);
				this.km.addKey(newkey);
			}
		}
	}
	print("Copy complete.");
}



ClonePlugin.prototype.toString = function() {
	return "Clone-Plugin";
}
