/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2025 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview KeyEscrowDAO
 */

var KeyEscrow = require('scsh/pki-db/KeyEscrow').KeyEscrow;



 /**
 * Data access object for key escrows
 *
 * @param {Object} the factory that created this DAO
 */
function KeyEscrowDAO(factory) {
	GPSystem.log(GPSystem.DEBUG, module.id, "new()");

	this.factory = factory;
}

exports.KeyEscrowDAO = KeyEscrowDAO;



KeyEscrowDAO.create = [
"CREATE TABLE IF NOT EXISTS KeyEscrow (" +
"	id MEDIUMINT UNSIGNED NOT NULL AUTO_INCREMENT," +
"	signerId MEDIUMINT UNSIGNED NOT NULL," +
"	keyId VARBINARY(64) NOT NULL," +
"	name VARCHAR(100) NOT NULL," +
"	exportCertificateChain VARBINARY(2700) NOT NULL," +
"	keyblob VARBINARY(8192) NOT NULL," +
"	PRIMARY KEY (id)," +
"	FOREIGN KEY (signerId) REFERENCES Signer(id) ON DELETE CASCADE" +
")",
{	H2: "CREATE INDEX IF NOT EXISTS KeyEscrow_Idx1 ON KeyEscrow(keyId);",
	MySQL: {
		IsEmpty: "SHOW INDEX FROM KeyEscrow WHERE Key_name = 'KeyEscrow_Idx1';",
		Statement: "ALTER TABLE KeyEscrow ADD INDEX KeyEscrow_Idx1 (keyId);"
	}}
];

KeyEscrowDAO.drop = "DROP TABLE IF EXISTS KeyEscrow";



KeyEscrowDAO.prototype.toString = function() {
	return "KeyEscrowDAO(db)";
}



/**
 * Get a list of keys in escrow for the given signer
 *
 * @param {Number} signerId the signerId of the escrow key
 * @type KeyEscrow[]
 * @return the list of keys in escrow
 */
KeyEscrowDAO.prototype.getKeyEscrowBySigner = function(signerId) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getKeyEscrowBySigner(" + signerId + ")");

	assert(typeof(signerId) == "number", "signerId must be number");

	var con = null;
	var stmt = null;
	var rs = null;

	var list = [];

	try	{
		con = this.factory.getConnection();

		stmt = con.prepareStatement("SELECT * FROM KeyEscrow WHERE signerId = ?;");
		stmt.setInt(1, signerId);

		rs = stmt.executeQuery();
		while (rs.next()) {
			var keyEscrow = new KeyEscrow(this);
			this.factory.resultSetToProperties(rs, keyEscrow);
			list.push(keyEscrow);
		}
	}
	finally {
		if (rs != null) {
			rs.close();
		}
		if (stmt != null) {
			stmt.close();
		}
		if (con != null) {
			con.close();
		}
	}

	return list;
}
